# Apptorium Code Formatting Rules

The rules below describe Swift code formatting aligned with the style
used in AptApp and AptUtilities. Apply them across all Apptorium modules.

## 1. Indentation and Whitespace
- Indentation: 4 spaces, no tabs.
- Use spaces around binary operators (`=`, `==`, `+`, `&&`, etc.).
- Use a space after commas in argument lists and collections.
- Do not use blank lines "for decoration". Use them only to separate
  logical blocks.
- Maximum line length: **80 characters**. If a line is longer, wrap it
  according to sections 5-7.

## 2. Braces and Blocks (Critical)
- Opening brace stays on the same line as the declaration or control
  statement (K&R style).
- Closing brace is always on its own line.
- `else` and `catch` stay on the same line as the closing brace of the
  previous block.
- **Single-line blocks are strictly forbidden.** This applies to
  `guard`, `if`, `else`, `while`, `for`, `do`, `switch` (block), `defer`,
  and `closure` blocks.

NO:
```swift
guard condition else { return nil }
```

YES:
```swift
guard condition else {
    return nil
}
```

## 3. Blank Lines Between `{}` Blocks (Required)
- **Always** separate adjacent `{}` blocks with one blank line.
- This applies to sequences like
  `if`/`for`/`while`/`guard`/`switch`/`do`/`defer` and any block-level
  statements inside a function.
- Do **not** add a blank line between `}` and `else`/`catch` (they are
  one logical block).

Example:
```swift
func aa() {
    if conditionA {
        // ...
    }

    if conditionB {
        // ...
    }

    aaa()
    bbbb()

    if conditionC {
        // ...
    }
}
```

## 4. `guard` and Early Exits
- `guard` must always be multiline.
- `return`/`throw`/`break` must always be on their own line.
- Leave one blank line after a `guard` block before continuing logic.
- Group preliminary conditions in a single `guard` when they represent
  a shared early-exit check.

## 5. Conditionals and Loops
- For multiline conditions, break after logical operators.
- For complex conditions, prefer line wrapping over one very long line.
- Do not use unary negation (`!`) in conditions.
- Use explicit boolean comparisons instead:
  `if condition == false` instead of `if !condition`.

Example:
```swift
if isReady &&
   isAuthorized &&
   hasNetwork {
    // ...
}
```

## 6. Line Wrapping and Parameter Lists
- Keep function calls on one line when they fit within the 80 character
  limit. Only wrap when they exceed the limit.
- If a parameter list does not fit in one line, put each parameter on
  its own line.
- Closing parenthesis goes on its own line, with no trailing comma after
  the last argument.
- This applies to both function/initializer declarations and function calls.

Example:
```swift
public init(
    expiryDate: Date,
    downloadURL: URL,
    checkingInterval: TimeInterval = 3600
) {
    self.expiryDate = expiryDate
    self.downloadURL = downloadURL
    self.interval = checkingInterval
}
```

## 7. Call Chains (SwiftUI and Others)
- Write long method chains in cascading form, one method per line.
- Put the dot at the beginning of the next line.

Example:
```swift
Text(message)
    .font(.callout.weight(.semibold))
    .multilineTextAlignment(.center)
```

## 8. `switch`
- `case` and `default` go on separate lines.
- Short statements after `:` may stay on one line (without extra
  braces), for example `case .export: self = .exporting`.
- For more complex logic, use a full multiline block.

## 9. Collections and Literals
- In multiline collections, use one element per line.
- Preferred dictionary type style: `[Key: Value]` (no spaces around `:`).
- In maps and filters, prefer explicit names like `key` and `value`
  instead of unclear abbreviations.

## 10. Attributes and Compiler Directives
- Attributes (`@available`, `@objc`, `@IBAction`) go on separate lines
  above the declaration.
- Compiler directives (`#if os(...)`) go on separate lines and should be
  surrounded by blank lines when they separate sections.

## 11. File Layout
- Keep the Xcode file header comment at the top.
- Keep `import` statements without blank lines between consecutive imports.
- Section order inside a type: properties -> init -> public methods ->
  private methods -> helper types.

## 12. Comments and Documentation
- Use `///` doc comments for public APIs.
- Use block comments only when truly needed. Prefer short, factual
  descriptions.

## 13. Readability and Naming
- **Do not use abbreviations in names.**
  Names must be complete and understandable.
- Prefer descriptive variable, function, and type names over short
  one-word shortcuts.
- Single-letter names are allowed only in very short, local contexts
  (for example simple loops) when meaning is obvious.

## 14. Code Structure
- Code must be readable and split into meaningful units: functions,
  classes, structs, extensions.
- Functions should have one responsibility and stay as short as possible.
- Break larger logic blocks into private helper methods.
- Every struct, class, and protocol should be in a separate file.
- Directories: `Views` (with proper sections, for example Main Window,
  Settings, and logical subfolders; if a view needs its own subviews, a
  local `Subviews` folder can be placed there), `Data`, `Core`.
